import { useEffect } from 'react';
import safeHasOwnProperty from './shared/safeHasOwnProperty';
import useEvent from './useEvent';
/**
 * Returns an object of callback setters to handle the drag-related events.
 * It accepts a DOM ref representing the events target (where attach the events to).
 *
 * Returned callback setters: `onDrag`, `onDrop`, `onDragEnter`, `onDragEnd`, `onDragExit`, `onDragLeave`,
 * `onDragOver`, `onDragStart`;
 */
const useDragEvents = (targetRef, isDraggable = true) => {
    const onDrag = useEvent(targetRef, 'drag');
    const onDrop = useEvent(targetRef, 'drop');
    const onDragEnter = useEvent(targetRef, 'dragenter');
    const onDragEnd = useEvent(targetRef, 'dragend');
    const onDragExit = useEvent(targetRef, 'dragexit');
    const onDragLeave = useEvent(targetRef, 'dragleave');
    const onDragOver = useEvent(targetRef, 'dragover');
    const onDragStart = useEvent(targetRef, 'dragstart');
    if (targetRef !== null && !safeHasOwnProperty(targetRef, 'current')) {
        throw new Error('Unable to assign any drag event to the given ref');
    }
    useEffect(() => {
        if (isDraggable && targetRef.current && !targetRef.current.hasAttribute('draggable')) {
            targetRef.current.setAttribute('draggable', String(true));
        }
    }, []);
    return Object.freeze({
        onDrag,
        onDrop,
        onDragEnter,
        onDragEnd,
        onDragExit,
        onDragLeave,
        onDragOver,
        onDragStart
    });
};
export default useDragEvents;
